/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize
#define iMouse vec4(0.0,0.0, 0.0,0.0)

/////////////////////////////////////////////////////////////////////////////////

// The ShaderToy shaders often use textures as inputs named iChannel0. With VGHD
// this may access a Sprite, ClipSprite or ClipNameSprite image depending on how
// the .scn file declares them.
//
// Note, the name used here does not seem to make any difference, so I have used
// iChannel0 which is what is used by ShaderToy but you can use any name as long
// as it matches the use in the main body of the shader. TheEmu.

uniform sampler2D iChannel0;
uniform sampler2D iChannel1;

// With VGHD the range of the P argument's components of the texture functions is
// 0.0 to 1.0 whereas with ShaderToy it seems that the upper limits are given  by
// the number of pixels in each direction, typically 512 or 64.  We therefore use
// the following functions instead.

vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}

// Rather than edit the body of the original shader we use use a define  here  to
// redirect texture calls to the above functions.

#define texture2D texture2D_Fract

/////////////////////////////////////////////////////////////////////////////////

mat3 rotx(float a) { mat3 rot; rot[0] = vec3(1.0, 0.0, 0.0); rot[1] = vec3(0.0, cos(a), -sin(a)); rot[2] = vec3(0.0, sin(a), cos(a)); return rot; }
mat3 roty(float a) { mat3 rot; rot[0] = vec3(cos(a), 0.0, sin(a)); rot[1] = vec3(0.0, 1.0, 0.0); rot[2] = vec3(-sin(a), 0.0, cos(a)); return rot; }
mat3 rotz(float a) { mat3 rot; rot[0] = vec3(cos(a), -sin(a), 0.0); rot[1] = vec3(sin(a), cos(a), 0.0); rot[2] = vec3(0.0, 0.0, 1.0); return rot; }

const float WLEN = 8.0;
const float phase = 2.2;


struct Ray
{
    vec3 o;
    vec3 p;
    vec3 d;
};

    
void filter(inout vec4 col, float f)
{
    col -= f;
    col = clamp(col, 0.0, 1.0);
    col /= 1.0 - f;
}

float filter(float f, float a)
{
    f -= a;
    f = clamp(f, 0.0, 1.0);
    f /= 1.0 - a;
    return f;
}
    
float wv(in vec2 uv, vec2 d, float t, float A)
{
    
    float h1 = (sin ( dot(d, uv) / WLEN + phase * t)) * 0.5;
    
    float h2 = (sin ( dot(d, uv) / WLEN + phase * t));
    h2 *= h2;
    h2 *= 0.2;
    
    float h3 = (sin ( dot(d, uv) / WLEN + phase * t));
    h3 *= h3 * h3;
    h3 *= 0.1;
        
    float w = sin(dot(uv.yx * .015, d)) * 0.5 + 0.5; 
    return mix(w, 1.0, 0.9) * (h1 + h2 + h3) * A;
}

float waves(in vec2 uv, float t, float A)
{
    float w1 = wv(uv, vec2(1.0, -0.1), t, A) * 0.4;
    float w2 = wv(uv * 14.0, vec2(0.5, 0.2), t, A) * 0.015;
    
    vec2 d2 = vec2(0.4, 0.7);
    float w3 = wv(uv * 2.0, d2, t * 1.5, A) * 0.25;

    vec2 d3 = (vec2(0.7, 0.5));
    float w4 = wv(uv * 5.0, d3, t * 1.5, A) * 0.15;
    float w5 = wv(uv * 14.0, vec2(0.0, 0.5), t, A) * 0.015;
    
    return w5 + w4 + w3 + w2 + w1;
}

float map(in vec3 rp)
{
    
    vec2 uv = rp.xz * 0.08;
    float b = texture2D(iChannel0, uv).r * 0.5;
    b += texture2D(iChannel0, uv * 2.0).r * 0.25;
    b += texture2D(iChannel0, uv * 4.0).r * 0.0125;
    b *= 0.025;
    // choppy waves by offsetting ray
    rp.x += pow(max(rp.y, 0.0), 2.3) * 0.45;
    
    float A = 4.0;
    float w = rp.y - waves(rp.xz, 2.0 + iGlobalTime * 0.5, A);
    return w + b;
}

vec3 grad(vec3 rp)
{
    vec2 off = vec2(0.001, 0.0);
    vec3 g = vec3(map(rp + off.xyy) - map(rp - off.xyy),
				  map(rp + off.yxy) - map(rp - off.yxy),                  
				  map(rp + off.yyx) - map(rp - off.yyx));
    return normalize(g);
}

vec3 light = normalize(vec3(-0.2, 0.1, -1.0));


bool trace(inout Ray r, inout vec4 color)
{
    vec3 rd = r.d;
    float topy = 2.8;
    r.p += r.d * (r.p.y - topy) / abs(r.d.y);
    
    float dist = 0.0;
    
	for (int i = 0; i < 80; ++i)
    {
        float h = map(r.p);
        if ( h < 0.7)
        {
		    vec3 rp = r.p;
            vec3 g = grad(r.p);
            // diffuse
            float d = dot(g, light);
            d = clamp(d, 0.1, 1.0) * 0.4;
            color = vec4(0.6, 0.85, 0.9, 1.0) * d;
            
            // rim
            float r = 1.0 - clamp(dot(-r.d, g), 0.0, 1.0);
            
            r = pow(r, 4.0);
            color += vec4(0.6, 0.8, 0.95, 1.0) * r * 0.6;
            
            // static foam
            vec4 wcol = texture2D(iChannel1, rp.xz * 0.003);
            wcol = vec4(wcol.r);
            filter(wcol, 0.7);
            color += wcol * 0.6;
            
            // dynamic foam
            vec4 topfoam = texture2D(iChannel1, rp.xz * 0.02).rrrr;
            vec4 topfoam2 = texture2D(iChannel1, (rp.zy +vec2(0.0, -iGlobalTime * 5.0)) * 0.05).rrrr;
            
            topfoam = mix(topfoam, topfoam2, dot(g, normalize(vec3(-1.0, -0.0, 0.5))));
            topfoam = clamp(topfoam, 0.0, 1.0);
			float f = smoothstep(2.0, 2.4, rp.y);
            topfoam = mix(vec4(0.0), topfoam, f) * 0.5;
            color += topfoam;
            
            // spec
            vec3 H = normalize(-rd + normalize(vec3(0.0, 2.0, 9.0)));
            float spc = dot(H, g);
            spc = clamp(spc, 0.0, 1.0);
            spc = pow(spc, 30.0);
            
            color += vec4(0.7, 0.7, 0.6, 1.0) * spc * 0.6;
            return true;
        }
        float trvl = max(h * (0.25 * exp(max(0.0, dist - 5.0) * .07)), 0.01); 
        r.p += r.d * trvl;
        dist += trvl;
        if(dist > 180.0)
        {
            return false;
        }
    }
    return false;
}

float fbm (vec2 uv)
{
    float f = (texture2D(iChannel0, uv).r - 0.5) * 0.5;
    f += (texture2D(iChannel0, uv * 2.0).r - 0.5) * 0.25;
    f += (texture2D(iChannel0, uv * 4.0).r - 0.5) * 0.125;
    f += (texture2D(iChannel0, uv *  8.0).r - 0.5) * 0.125 * 0.5;
    f += (texture2D(iChannel0, uv * 16.0).r - 0.5) * 0.125 * 0.5 * 0.5;
    f += (texture2D(iChannel0, uv * 32.0).r - 0.5) * 0.125 * 0.5 * 0.5 * 0.5;
    return clamp(f + 0.5, 0.0, 1.0);
}

void mainImage( out vec4 fragColor, in vec2 fragCoord )
{
	vec2 uv = fragCoord.xy / iResolution.xy;
    uv -= vec2(0.5);
    uv.y /= iResolution.x / iResolution.y;
    fragColor = vec4(1.0);
    
    // movement
    vec3 _uv = vec3(uv, 0.0);
    _uv.y += sin(iGlobalTime * 0.5) * 0.05;
    _uv *= rotz(sin(iGlobalTime * 0.5) * 0.1);
    
    uv = vec2(_uv);
    
    // tracing
    Ray r;
    r.o = vec3(0.0, 10.0, -1.0 + iGlobalTime * 5.0);
    r.p = r.o;
    r.d = normalize(vec3(uv + vec2(0.0, -0.1), 1.));
    
    vec3 ro = r.o;
    bool hit = trace(r, fragColor);
    
    // bg fog
    float l = length(ro - r.p);
    float s = smoothstep(70.0, 170.0, l);
    vec4 fog = vec4(0.3, 0.7, 0.95, 1.0);
    float fg = smoothstep(-0.3, 0.3, uv.y);
    fog = mix(fog, vec4(0.2), fg);
    fragColor = mix(fragColor, fog,  s);

    
	// out of gamut correction    
    float m = max(fragColor.r, fragColor.g);
    m = max(fragColor.b, m);
    m /= max(1.0, m);
    
    // clouds
    vec2 skyuv = vec2(r.d.x / r.d.y, r.d.z / r.d.y) * 0.005;
    float t = iGlobalTime * .012;
    
    float f = fbm(skyuv + vec2(0.0, t));
    f = filter(f, 0.5) * 0.5;

    float f2 = fbm(skyuv * 1.5 + vec2(0.0, t * 1.1));
    f += filter(f2, 0.4) * 0.25;

    float f3 = fbm(skyuv * 2.5 + vec2(0.0, t * 1.4));
    f += filter(f2, 0.2) * 0.0125;
    f = clamp(f, 0.0, 1.0) * smoothstep(0.05, 0.5, uv.y);
    
    vec4 sky = vec4(1.0) * f;
    
    float moon = 1.0 - smoothstep(0.04, 0.05, length(uv - vec2(0.0, 0.2)));
    float halo = 1.0 - smoothstep(0.04, 0.4, length(uv - vec2(0.0, 0.2)));
    
    vec4 mooncol = moon * mix(texture2D(iChannel1, uv * 2.0).rrrr, vec4(1.0), 0.3);
    mooncol += halo * vec4(1.0) * 0.2;
    mooncol = clamp(mooncol, 0.0, 1.0);
    
    fragColor += mix(mooncol, sky, clamp(f * 15.0, 0.0, 1.0));
	
}

void main ( void )
{
  mainImage( gl_FragColor, gl_FragCoord.xy );
}